<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';

// Set active page for navigation
$activePage = 'suppliers';

// Check if user is logged in
requireLogin();

// Check if user has permission to view suppliers
// Check if user has any supplier-related permissions
$supplierPermissions = [
    'view_suppliers', 'manage_suppliers', 'add_supplier', 'edit_supplier', 'delete_supplier',
    'search_suppliers', 'process_supplier_payment', 'add_supplier_item'
];

if (!hasAnyPermission($supplierPermissions)) {
    $_SESSION['error'] = "You don't have permission to view suppliers.";
    header('Location: dashboard.php');
    exit;
}

if (false && !hasPermission('view_suppliers')) { // Disable the old check
    $_SESSION['error'] = "You don't have permission to access the Suppliers page.";
    header('Location: ../pages/dashboard.php');
    exit;
}

// Get all suppliers
$stmt = $pdo->query("
    SELECT 
        s.*,
        COUNT(DISTINCT sr.id) as total_products,
        SUM(sr.quantity) as total_stock,
        SUM(sr.quantity * sr.unit_price) as total_value
    FROM suppliers s
    LEFT JOIN supplier_receipts sr ON s.id = sr.supplier_id
    GROUP BY s.id
    ORDER BY s.name
");
$suppliers = $stmt->fetchAll();

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh';

// Get store information
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings");
$stmt->execute();
$storeSettings = [];
while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $storeSettings[$row['setting_key']] = $row['setting_value'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>View Suppliers - EDDUH COMPUTERS</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        @media print {
            body * {
                visibility: hidden;
            }
            .print-section, .print-section * {
                visibility: visible;
            }
            .print-section {
                position: absolute;
                left: 0;
                top: 0;
                width: 100%;
            }
            .no-print {
                display: none !important;
            }
            .print-header {
                text-align: center;
                margin-bottom: 1rem;
                padding-bottom: 0.5rem;
                border-bottom: 1px solid #e2e8f0;
            }
            .print-header h1 {
                font-size: 1.5rem;
                margin-bottom: 0.25rem;
            }
            .print-header p {
                font-size: 0.75rem;
                margin-bottom: 0.25rem;
            }
            .print-table {
                width: 100%;
                border-collapse: collapse;
                margin-bottom: 0.75rem;
                font-size: 0.7rem;
            }
            .print-table th, .print-table td {
                padding: 0.25rem 0.5rem;
                border: 1px solid #e2e8f0;
            }
            .print-table th {
                background: #f8fafc;
                font-weight: 600;
            }
            @page {
                margin: 0.5cm;
            }
        }

        /* Modern Button Styles */
        .btn-action {
            padding: 0.4rem 0.6rem;
            border-radius: 6px;
            transition: all 0.2s ease;
            margin: 0 2px;
        }

        .btn-action:hover {
            transform: translateY(-1px);
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        .btn-action i {
            font-size: 0.9rem;
        }

        .btn-view {
            background-color: #3b82f6;
            color: white;
        }

        .btn-view:hover {
            background-color: #2563eb;
            color: white;
        }

        .btn-edit {
            background-color: #10b981;
            color: white;
        }

        .btn-edit:hover {
            background-color: #059669;
            color: white;
        }

        .btn-delete {
            background-color: #ef4444;
            color: white;
        }

        .btn-delete:hover {
            background-color: #dc2626;
            color: white;
        }

        /* Modal Styles */
        .modal-content {
            border-radius: 12px;
            border: none;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
        }

        .modal-header {
            border-bottom: 1px solid #e5e7eb;
            padding: 1rem 1.5rem;
        }

        .modal-body {
            padding: 1.5rem;
        }

        .modal-footer {
            border-top: 1px solid #e5e7eb;
            padding: 1rem 1.5rem;
        }

        .confirmation-icon {
            font-size: 3rem;
            margin-bottom: 1rem;
        }

        .confirmation-text {
            font-size: 1.1rem;
            color: #374151;
        }

        .confirmation-details {
            background-color: #f9fafb;
            border-radius: 8px;
            padding: 1rem;
            margin-top: 1rem;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>

                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h2 class="mb-1">View Suppliers</h2>
                        <p class="text-muted mb-0">Complete list of all suppliers</p>
                    </div>
                    <button type="button" class="btn btn-primary" onclick="printReport()">
                        <i class="fas fa-print me-2"></i>Print Report
                    </button>
                </div>

                <!-- Print Section -->
                <div class="print-section" style="display: none;">
                    <div class="print-header">
                        <h1><?php echo htmlspecialchars($storeSettings['store_name'] ?? 'Store Name'); ?></h1>
                        <p><?php echo htmlspecialchars($storeSettings['store_address'] ?? ''); ?></p>
                        <p>Phone: <?php echo htmlspecialchars($storeSettings['store_phone'] ?? ''); ?></p>
                        <p>Suppliers Report - <?php echo date('Y-m-d H:i:s'); ?></p>
                    </div>

                    <table class="print-table">
                        <thead>
                            <tr>
                                <th>Supplier Name</th>
                                <th>Contact Person</th>
                                <th>Phone</th>
                                <th>Email</th>
                                <th>Address</th>
                                <th>Products</th>
                                <th>Total Stock</th>
                                <th>Total Value</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($suppliers as $supplier): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($supplier['name']); ?></td>
                                <td><?php echo htmlspecialchars($supplier['contact_person']); ?></td>
                                <td><?php echo htmlspecialchars($supplier['phone']); ?></td>
                                <td><?php echo htmlspecialchars($supplier['email']); ?></td>
                                <td><?php echo htmlspecialchars($supplier['address']); ?></td>
                                <td><?php echo number_format($supplier['total_products']); ?></td>
                                <td><?php echo number_format($supplier['total_stock'] ?? 0); ?></td>
                                <td><?php echo $currency . ' ' . number_format($supplier['total_value'] ?? 0, 2); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Main Content -->
                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Supplier Name</th>
                                        <th>Contact Person</th>
                                        <th>Phone</th>
                                        <th>Email</th>
                                        <th>Address</th>
                                        <th>Products</th>
                                        <th>Total Stock</th>
                                        <th>Total Value</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($suppliers as $supplier): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($supplier['name']); ?></td>
                                        <td><?php echo htmlspecialchars($supplier['contact_person']); ?></td>
                                        <td><?php echo htmlspecialchars($supplier['phone']); ?></td>
                                        <td><?php echo htmlspecialchars($supplier['email']); ?></td>
                                        <td><?php echo htmlspecialchars($supplier['address']); ?></td>
                                        <td><?php echo number_format($supplier['total_products']); ?></td>
                                        <td><?php echo number_format($supplier['total_stock'] ?? 0); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($supplier['total_value'] ?? 0, 2); ?></td>
                                        <td>
                                            <?php if (hasPermission('manage_suppliers')): ?>
                                            <div class="btn-group">
                                                <a href="view_supplier_details.php?id=<?php echo $supplier['id']; ?>" 
                                                   class="btn btn-action btn-view" 
                                                   title="View Details">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                                <a href="edit_supplier.php?id=<?php echo $supplier['id']; ?>" 
                                                   class="btn btn-action btn-edit" 
                                                   title="Edit Supplier">
                                                    <i class="fas fa-edit"></i>
                                                </a>
                                                <button type="button" 
                                                        class="btn btn-action btn-delete" 
                                                        onclick="confirmDelete(<?php echo $supplier['id']; ?>, '<?php echo addslashes($supplier['name']); ?>')"
                                                        title="Delete Supplier">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </div>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteModal" tabindex="-1" aria-labelledby="deleteModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="deleteModalLabel">Confirm Delete</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body text-center">
                    <div class="confirmation-icon text-danger">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <p class="confirmation-text">Are you sure you want to delete this supplier?</p>
                    <div class="confirmation-details">
                        <p class="mb-0"><strong>Supplier Name:</strong> <span id="deleteSupplierName"></span></p>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-danger" id="confirmDeleteBtn">Delete</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let supplierToDelete = null;

        function confirmDelete(id, name) {
            supplierToDelete = id;
            document.getElementById('deleteSupplierName').textContent = name;
            new bootstrap.Modal(document.getElementById('deleteModal')).show();
        }

        document.getElementById('confirmDeleteBtn').addEventListener('click', function() {
            if (supplierToDelete) {
                window.location.href = `delete_supplier.php?id=${supplierToDelete}`;
            }
        });

        function printReport() {
            document.querySelector('.print-section').style.display = 'block';
            window.print();
            setTimeout(() => {
                document.querySelector('.print-section').style.display = 'none';
            }, 100);
        }
    </script>
    <?php include '../includes/footer.php'; ?>
</body>
</html> 